#include <mb_common.h>
#include <regex.h>

#ifndef MYNAME
#include "MYNAME is not defined in the Makefile"
#endif
#ifndef VERSION
#include "VERSION is not defined in the Makefile"
#endif
#define RECURSE	"recurse"

/* yeah, yeah, global variables are ugly ... */
static char	*checkhost = NULL;
static char	*cfgfile = MBDIVERT_CFG;
static char	*divert = NULL;
static int	pluginmatches = TRUE;	/* anything matches by default */
static int	forcedirect = FALSE;
static int	method = METH_NRPE;
static int	usesyslog = FALSE;
static int	syslogopened = FALSE;
static int	tagoutput = FALSE;
static int	unknownhostfail = FALSE;
static int	verbose = FALSE;
static char	*plugin = NULL;		/* point to plugin name */
static char	*nrpe_cmd = NULL;
static char	*nrpe_port = NULL;
static char	*nrpe_socketfail_unknown = NULL;
static char	*nrpe_timeout = NULL;
static char	*ssh_cmd = NULL;
static char	*ssh_identity = NULL;
static char	*ssh_login = NULL;
static char	*ssh_port = NULL;
static char	*ssh_userknownhostsfile = NULL;


static void
usage()
{
    mprintf( "usage: %s [-c cfgfile] [-d divert] [-H hostname]", _argv0 );
    mprintf( "    [-f] [-l] [-n] [-s] [-t] [-u] [-v] plugin args ..." );
    mprintf( "default cfgfile is %s", MBDIVERT_CFG );
    mprintf( "command for NRPE is %s", NRPE_COMMAND );
    mprintf( "command for SSH is %s", SSH_COMMAND );
    msgdie( "program version %s", VERSION );
}

void
opensyslog() {
    if ( ! syslogopened ) {
	openlog( _argv0, LOG_PID, LOG_USER );
	syslogopened = TRUE;
    }
}

void /*VARARGS1*/
problem( const char *s, ... )
{
    va_list ap;
    va_start( ap, s );
    if ( verbose ) {
	/* we use stdout not stderr, so it shows up in nagios' display */
	fprintf( stdout, "%s: ", _argv0 );
	vfprintf( stdout, s, ap );
	fprintf( stdout, "\n" );
    }
    if ( usesyslog ) {
	opensyslog();
	vsyslog( LOG_ERR, s, ap );
    }
    va_end( ap );
}

void /*VARARGS1*/
inform( const char *s, ... )
{
    va_list ap;
    va_start( ap, s );
    if ( verbose ) {
	fprintf( stdout, "%s: ", _argv0 );
	vfprintf( stdout, s, ap );
	fprintf( stdout, "\n" );
    }
    if ( usesyslog ) {
	opensyslog();
	vsyslog( LOG_INFO, s, ap );
    }
    va_end( ap );
}



char *
mystrdup( char **p, char *s )
{
    if ( *p )
	free( *p );
    *p = strdup( s );
    if ( !*p )
	msgdie( "could not strdup a string of length %d: %s",
	    strlen( s ), strerror( errno ) );
    return( *p );
}


int
regexmatch( char *cfgfile, int linenum, char *r, char *s )
{
    char errbuf[512];
    regex_t re;
    int res;

    res = regcomp( &re, r, REG_EXTENDED|REG_NOSUB|REG_ICASE );
    if ( res != 0 ) {
	regerror( res, &re, errbuf, sizeof(errbuf) );
	msgdie( "config file %s line %d: regular expression error in '%s': %s",
	    cfgfile, linenum, r, errbuf );
    }
    res = regexec( &re, s, (size_t) 0, NULL, 0 );
    regfree( &re );
    return( res == 0 );
}


/* look for the host in the cfgfile and figure out what to do */
static void
lookup()
{
    char buf[1024], *key, *arg;
    FILE *fp;
    int linenum = 0;

    if ( ( fp = fopen( cfgfile, "r" ) ) == NULL )
	msgdie( "could not open config file: %s: %s",
	    cfgfile, strerror( errno ) );
    /*
     * nrpe pathname/to/check_nrpe
     * ssh pathname/to/check_by_ssh
     * method nrpe|ssh
     * divert hostname
     * [!]host hostname
     * [!]prefix hostnamepart
     * [!]suffix hostnamepart
     * [!]match regex
     * path	value:for:PATH true-or-false
     * tagoutput true-or-false
     * unknownhostfail true-or-false
     */
    while ( fgets( buf, sizeof(buf), fp ) != NULL ) {
	linenum++;
	key = strtok( buf, " \t\r\n" );
	arg = strtok( NULL, " \t\r\n" );
	if ( !key || *key=='\0' || *key=='#' ) {
	    continue;		/* empty or comment line */
	}
	if ( strcasecmp( key, "divert" ) == 0 ) {
	    if ( !arg || *arg=='\0' ) {
		mystrdup( &divert, "" );
	    } else if ( strcmp( arg, "-" ) == 0 ) {
		/* divert to the checkhost itself */
		mystrdup( &divert, checkhost );
	    } else {
		mystrdup( &divert, arg );
	    }
	    continue;
	}
	if ( strcasecmp( key, "pluginmatch" ) == 0 ) {
	    if ( !arg || *arg=='\0' ) {
		/* empty means anything matches */
		pluginmatches = TRUE;
	    } else {
		pluginmatches = regexmatch( cfgfile, linenum, arg, plugin );
	    }
	    continue;
	}
	/* every other known keyword needs an argument */
	if ( !arg || *arg=='\0' ) {
	    msgdie( "config file %s line %d: keyword '%s' has no arg",
		cfgfile, linenum, key );
	}
	if ( strcasecmp( key, "nrpe" ) == 0 ) {
	    mystrdup( &nrpe_cmd, arg );
	} else if ( strcasecmp( key, "nrpe_timeout" ) == 0 ) {
	    mystrdup( &nrpe_timeout, arg );
	} else if ( strcasecmp( key, "ssh" ) == 0 ) {
	    mystrdup( &ssh_cmd, arg );
	} else if ( strcasecmp( key, "ssh_identity" ) == 0 ) {
	    mystrdup( &ssh_identity, arg );
	} else if ( strcasecmp( key, "ssh_login" ) == 0 ) {
	    mystrdup( &ssh_login, arg );
	} else if ( strcasecmp( key, "ssh_port" ) == 0 ) {
	    mystrdup( &ssh_port, arg );
	} else if ( strcasecmp( key, "ssh_userknownhostsfile" ) == 0 ) {
	    mystrdup( &ssh_userknownhostsfile, arg );
	} else if ( strcasecmp( key, "method" ) == 0 ) {
	    if ( strcasecmp( arg, "nrpe" ) == 0 )
		method = METH_NRPE;
	    else if ( strcasecmp( arg, "ssh" ) == 0 )
		method = METH_SSH;
	    else
		msgdie( "config file %s line %d: method type '%s' not known",
		    cfgfile, linenum, arg );
	} else if ( strcasecmp( key, "path" ) == 0 ) {
	    if ( setenv( "PATH", arg, TRUE ) != 0 ) {
		msgdie( "could not setenv PATH to a string of length %d: %s",
		    strlen( arg ), strerror( errno ) );
	    }
	} else if ( strcasecmp( key, "tagoutput" ) == 0 ) {
	    if ( strcasecmp( arg, "true" ) == 0 )
		tagoutput = TRUE;
	    else if ( strcasecmp( arg, "false" ) == 0 )
		tagoutput = FALSE;
	    else
		msgdie( "config file %s line %d: tagoutput value '%s' not known",
		    cfgfile, linenum, arg );
	} else if ( strcasecmp( key, "usesyslog" ) == 0 ) {
	    if ( strcasecmp( arg, "true" ) == 0 )
		usesyslog = TRUE;
	    else if ( strcasecmp( arg, "false" ) == 0 )
		usesyslog = FALSE;
	    else
		msgdie( "config file %s line %d: usesyslog value '%s' not known",
		    cfgfile, linenum, arg );
	} else if ( strcasecmp( key, "unknownhostfail" ) == 0 ) {
	    if ( strcasecmp( arg, "true" ) == 0 )
		unknownhostfail = TRUE;
	    else if ( strcasecmp( arg, "false" ) == 0 )
		unknownhostfail = FALSE;
	    else
		msgdie( "config file %s line %d: unknownhostfail value '%s' not known",
		    cfgfile, linenum, arg );
	} else if ( strcasecmp( key, "host" ) == 0 ) {
	    if ( pluginmatches && strcasecmp( arg, checkhost ) == 0 ) {
		break;		/* found a match */
	    }
	} else if ( strcasecmp( key, "!host" ) == 0 ) {
	    if ( pluginmatches && strcasecmp( arg, checkhost ) == 0 ) {
		forcedirect = TRUE;
		break;		/* found a match */
	    }
	} else if ( strcasecmp( key, "prefix" ) == 0 ) {
	    if ( pluginmatches && strcasestr( checkhost, arg ) == checkhost ) {
		break;		/* found a match */
	    }
	} else if ( strcasecmp( key, "!prefix" ) == 0 ) {
	    if ( pluginmatches && strcasestr( checkhost, arg ) == checkhost ) {
		forcedirect = TRUE;
		break;		/* found a match */
	    }
	} else if ( strcasecmp( key, "suffix" ) == 0 ) {
	    /* too bad there's no strrstr() */
	    int lh, ls;
	    lh = strlen( checkhost );
	    ls = strlen( arg );
	    if ( lh >= ls ) {	/* could be possible */
		if ( pluginmatches
		    && strcasecmp( checkhost+(lh-ls), arg ) == 0 ) {
		    break;	/* a match! */
		}
	    }
	} else if ( strcasecmp( key, "!suffix" ) == 0 ) {
	    /* too bad there's no strrstr() */
	    int lh, ls;
	    lh = strlen( checkhost );
	    ls = strlen( arg );
	    if ( lh >= ls ) {	/* could be possible */
		if ( pluginmatches
		    && strcasecmp( checkhost+(lh-ls), arg ) == 0 ) {
		    forcedirect = TRUE;
		    break;	/* a match! */
		}
	    }
	} else if ( strcasecmp( key, "match" ) == 0 ) {
	    if ( pluginmatches
		&& regexmatch( cfgfile, linenum, arg, checkhost ) ) {
		break;		/* a match! */
	    }
	} else if ( strcasecmp( key, "!match" ) == 0 ) {
	    if ( pluginmatches
		&&  regexmatch( cfgfile, linenum, arg, checkhost ) ) {
		forcedirect = TRUE;
		break;		/* a match! */
	    }
	} else {
	    msgdie( "config file %s line %d: unknown keyword '%s'",
		cfgfile, linenum, key );
	}
    }
    if ( feof( fp ) ) {
	/* got to the end of the config file with no host match */
	mystrdup( &divert, "" );	/* go direct, no diversion */
    }
    fclose( fp );
}


/* create a new argv for calling nrpe */
char **
nrpe_argv( argc, argv )
int argc;
char *argv[];
{
    /* 20 - gigantic freaking hack to avoid counting */
    char **p, **newargv = calloc( argc+20, sizeof( char * ) );
    int i;
    if ( ! newargv )
	msgdie( "could not calloc space for %d char *: %s",
	    argc+20, strerror( errno ) );
    p = newargv;
    *p++ = nrpe_cmd;
    *p++ = "-H";
    *p++ = divert;
    if ( nrpe_port ) {
	*p++ = "-p";
	*p++ = nrpe_port;
    }
    if ( nrpe_socketfail_unknown )
	*p++ = "-u";
    if ( nrpe_timeout ) {
	*p++ = "-t";
	*p++ = nrpe_timeout;
    }
    *p++ = "-c";
    /* send only basename of the plugin name */
    *p++ = basename( argv[0] );
    if ( argc > 1 ) {
	*p++ = "-a";
	for ( i=1; i<argc; i++ ) {
	    *p++ = argv[i];
	}
    }
    *p = NULL;
    return( newargv );
}


/* create a new argv for calling ssh */
char **
ssh_argv( argc, argv )
int argc;
char *argv[];
{
    /* 20 - gigantic freaking hack to avoid counting */
    char **p, **newargv = calloc( argc+20, sizeof( char * ) );
    int i;
    if ( ! newargv )
	msgdie( "could not calloc space for %d char *: %s",
	    argc+20, strerror( errno ) );
    p = newargv;
    *p++ = ssh_cmd;
    *p++ = "-n";
    *p++ = "-x";
    *p++ = "-e";
    *p++ = "none";
    if ( ssh_identity ) {
	*p++ = "-i";
	*p++ = ssh_identity;
    }
    if ( ssh_login ) {
	*p++ = "-l";
	*p++ = ssh_login;
    }
    if ( ssh_port ) {
	*p++ = "-p";
	*p++ = ssh_port;
    }
    if ( ssh_userknownhostsfile ) {
	*p++ = "-o";
	if ( asprintf( p, "UserKnownHostsFile=%s", ssh_userknownhostsfile )
	    == -1 ) {
	    msgdie( "could not asprintf a string: %s", strerror( errno ) );
	}
	p++;
    }
    *p++ = divert;
    for ( i=0; i<argc; i++ ) {
	*p++ = argv[i];
    }
    *p = NULL;
    return( newargv );
}


static void
exec_command( argv )
char **argv;
{
    inform( "running '%s' with some arguments (use -v if interested)",
	argv[0] );
    if ( verbose ) {
	/* this would be ugly to syslog, so I'll do it "later" */
	char **p = argv;
	fprintf( stderr, "%s: exec_command: args are:", _argv0 );
	for ( ; *p; p++ ) {
	    fprintf( stderr, " %s", *p );
	}
	fprintf( stderr, "\n" );
    }
    if ( tagoutput ) {
	if ( !forcedirect && ( divert && *divert ) ) {
	    fprintf( stdout, "divert: %s: ", divert );
	} else {
	    fprintf( stdout, "direct: " );
	}
	fflush( stdout );
    }
    execvp( argv[0], argv );
    msgdie( "could not execvp '%s': %s", argv[0], strerror( errno ) );
}


int
main( argc, argv )
int argc;
char *argv[];
{
    int c;

    errinit( argc, argv );

    /* help guard against exec() recursion */
    {
	char *v = getenv( MYNAME );
	if ( v && strcmp( v, RECURSE ) == 0 ) {
	    msgdie( "recursion blocked: environment %s=%s is set",
		MYNAME, RECURSE );
	}
    }
    (void) setenv( MYNAME, RECURSE, TRUE );
    /* stupid linux getopt() behaves wrong by default
     * if POSIXLY_CORRECT is set, it behaves right
     */
    (void) setenv( "POSIXLY_CORRECT", "yesplease", FALSE );

    mystrdup( &nrpe_cmd, NRPE_COMMAND );
    mystrdup( &ssh_cmd, SSH_COMMAND );

    if ( strcasecmp( _argv0, MYNAME ) == 0 ) {
	/* called by my own name */
	while ( (c = getopt( argc, argv, "c:d:fH:lnstuv" )) != -1 ) {
	    switch ( c ) {
		case 'c':
		    cfgfile = optarg;
		    break;
		case 'd':
		    divert = optarg;
		    break;
		case 'f':
		    forcedirect = TRUE;
		    break;
		case 'l':
		    usesyslog = TRUE;
		    break;
		case 'n':
		    method = METH_NRPE;
		    break;
		case 'H':
		    checkhost = optarg;
		    break;
		case 's':
		    method = METH_SSH;
		    break;
		case 't':
		    tagoutput = TRUE;
		    break;
		case 'u':
		    unknownhostfail = TRUE;
		    break;
		case 'v':
		    verbose = TRUE;
		    break;
		case 'h':
		case '?':
		default:
		    usage();
	    }
	}
	if ( optind >= argc ) {
	    usage();
	}
	plugin = argv[optind];
	argv += optind;
	argc -= optind;
    } else {
	/* called by the name of the desired plugin */
	plugin = _argv0;
	_argv0 = MYNAME;
    }

    inform( "argc is %d, optind is %d, plugin is '%s'",
	argc, optind, plugin );


    if ( ! checkhost ) {
	/* did not find a -H in our options (if any) */
	/* so look in the options given to the plugin and see what we find */
#ifdef HAVE_INT_OPTRESET
	/* do this if we can, otherwise I hope we're okay */
	optreset = 1;
#endif
	opterr = 0;	/* don't complain about what we don't recognize */
	optind = 1;
	inform("argc is %d, argv[0] is %s", argc, argv[0]);
	/* this only works if the -H is the *first* option with an arg */
	while ( (c = getopt( argc, argv, "H:" )) != -1 ) {
	    switch ( c ) {
		case 'H':
		    checkhost = optarg;
		    break;
		default:
		    /* ignore unknown opts */
		    break;
	    }
	}
    }

    if ( unknownhostfail && ! checkhost ) {
	problem( "don't know what host we're checking" );
	exit( EXIT_UNKNOWN );
    }

    inform( "checking host '%s'", checkhost ? checkhost : "UNKNOWN" );

    /* if we don't already know what to do, look it up */
    if ( checkhost && ( ! divert || method == METH_UNKNOWN ) ) {
	lookup();
    }

    if ( forcedirect || ! checkhost || ! divert || *divert=='\0' ) {
	/* no diversion - run the plugin directly */
	exec_command( argv );
    }

    if ( method == METH_NRPE ) {
	char **newargv = nrpe_argv( argc, argv );
	exec_command( newargv );
    }

    if ( method == METH_SSH ) {
	char **newargv = ssh_argv( argc, argv );
	exec_command( newargv );
    }

    msgdie( "don't know what to do - should not get here" );

    exit( 0 );
}
