#!/bin/sh

# nagios plugin for checking gvinum/vinum output on FreeBSD

# Likely remotely invoked by an snmpd exec or nrpe or check_by_ssh or ...
# Must run as root to get gvinum/vinum status

# Checks everything listed by vinum.
# The assumption is that if a volume has disappeared, it will no longer
# be mounted, and some other check will fail.
# What we do is check vinum output, and look for any problems.


PATH=/sbin:/usr/sbin:/bin:/usr/bin
export PATH

myname=`basename "$0"`
OK=0
WARNING=1
CRITICAL=2
UNKNOWN=3

tmp="/tmp/$myname.$$"
rm -f "$tmp"
trap "rm -f '$tmp'" EXIT

# vinum writes a log file by default, which grows and grows ...
VINUM_HISTORY=/dev/null
export VINUM_HISTORY

msg=""
str="OK"
res=$OK

# We'll assume that we've got the newer "gvinum" command unless we don't
vinum="gvinum"
if [ ! -f "/sbin/$vinum" ]; then
    vinum="vinum"
fi

echo "" > "$tmp"
errstr=`( $vinum list >> "$tmp" ) 2>&1`
if [ $? -ne 0 ]; then
    msg="${msg}$vinum failed: $errstr; "
    str="CRITICAL"
    res=$CRITICAL
fi

awkstr=`awk < "$tmp" '
    BEGIN { ok = 1; }
    NF == 0 { next; }
    /^[0-9][0-9]* (drives|volumes|plexes|subdisks):/ { next; }
    $1 == "D" {
	if ( $0 ~ /State: up/ ) {
	    drives++;
	} else {
	    printf " %s;", $0;
	    ok = 0;
	}
	next;
    }

    $1 == "P" {
	if ( $0 ~ /State: up/ ) {
	    plexes++;
	} else {
	    printf " %s;", $0;
	    ok = 0;
	}
	next;
    }

    $1 == "S" {
	if ( $0 ~ /State: up/ ) {
	    subdisks++;
	} else {
	    printf " %s;", $0;
	    ok = 0;
	}
	next;
    }

    $1 == "V" {
	if ( $0 ~ /State: up/ ) {
	    volumes++;
	} else {
	    printf " %s;", $0;
	    ok = 0;
	}
	next;
    }

    END {
	if ( volumes > 0 ) {
	    printf " %s volumes OK;", volumes;
	} else {
	    printf " no OK volumes;";
	    ok = 0;
	}
	if ( plexes > 0 ) {
	    printf " %s plexes OK;", plexes;
	} else {
	    printf " no OK plexes;";
	    ok = 0;
	}
	if ( subdisks > 0 ) {
	    printf " %s subdisks OK;", subdisks;
	} else {
	    printf " no OK subdisks;";
	    ok = 0;
	}
	if ( drives > 0 ) {
	    printf " %s drives OK;", drives;
	} else {
	    printf " no OK drives;";
	    ok = 0;
	}
	print "";	# the newline
	exit( ok ? 0 : 1 );
    }'`
if [ $? -ne 0 ]; then
    str="CRITICAL"
    res=$CRITICAL
fi
msg="${msg}$awkstr"

echo "${str}: $msg" | expand | sed -e 's/  */ /g'
exit $res
