#!/bin/sh

# This is a cover script for check_winsvc that maintains a list of
# services that we want checked and invokes check_winsvc with the
# appropriate args.

# Analogous to check_allstorage - gets a list the first time we
# see a windows box, and complains if any of those services disappear.

PATH=/usr/local/nagios/libexec:/usr/local/bin:/bin:/usr/bin
export PATH
umask 022

myname=`basename "$0"`
usage="usage: $myname host comm"

tmpout="/tmp/$myname.out.$$"
trap "rm -f $tmpout" EXIT
rm -f "$tmpout"


# Cache directory for what services to check for each host
# Can have files IGNORE and FORCE in addition to files named for hosts.
# IGNORE is used when we first create a service list for a host,
# and we exclude those services from the list we make and cache.
# FORCE is a list of services that we force a check of, or if the
# service name is prepended with a - we never check that, even if
# it is in the per-host file.
# Hash-comments are allowed.
# We assume no service name will ever start with a -
sdir=/tmp/check_winservices
mkdir -p "$sdir"

# nagios status codes
OK=0
WARNING=1
CRITICAL=2
UNKNOWN=3

cd "$sdir" || exit $UNKNOWN

if [ ! -f FORCE ]; then
    cat > FORCE <<EOF
# You can put a list of service names to force a check on here.
# Prepend with - to prevent a check.
EOF
fi
if [ ! -f IGNORE ]; then
    cat > IGNORE <<EOF
# You can put a list of service names to ignore when we first
# enumerate the services running on a host.
EOF
fi


if [ "$#" -ne 2 ]; then
    echo "$myname: $usage"
    exit $UNKNOWN
fi

host="$1"; shift
comm="$1"; shift

# if we don't already have a list of services for the host, create one
if [ ! -f "$host" ]; then
    check_winsvc "$host" "$comm" > "$tmpout"
    res=$?
    if [ "$res" -ne "$OK" ]; then
        cat "$tmpout"
        exit "$res"
    fi
    sed < IGNORE \
        -e '/^ *#/d' \
        -e '/^ *$/d' \
        -e 's/^ *//' \
        -e 's/ *$//' \
        -e 's/^/-/' \
        | cat - "$tmpout" \
        | awk -v host="$host" '
            /^-/ { sub( /^-/, "" ); block[$0] = "yes"; next; }
            {
                sub( /^.*: */, "" );
                split( $0, services, /; */ );
                for ( i in services ) {
                    name = services[i];
                    sub( / *installed *active *$/, "", name );
                    if ( name != "" && block[name] != "yes" ) {
                        print name >host;
                    }
                }
            }
            '
fi


# Now we make our list of services to feed into check_winsvc.
# We put it in a /tmp file so that we can get the exit code from
# check_winsvc easily, which is less obvious with a pipeline.
# Call expand first to change any tabs into spaces for easier parsing.
expand "$host" FORCE \
| awk -v quote="'" '
    { sub( /^ */, "" ); sub( / *$/, "" ); }
    /^ *#/ { next; }
    /^ *$/ { next; }
    /^-/ {
        sub( /^- */, "" );
        list[$0] = "no";
        next;
    }
    {
        list[$0] = "yes";
    }
    END {
        for ( name in list ) {
            if ( list[name] == "yes" ) {
                printf "%s%s%s\n", quote, name, quote;
            }
        }
    }
    ' \
    | sort > "$tmpout"

eval set -- `cat "$tmpout"`

# clean up
rm -f "$tmpout"

exec check_winsvc "$host" "$comm" "$@"
